#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/delay.h>
#include <asm/gpio.h>
#include <linux/earlysuspend.h>
#include <linux/semaphore.h>
#include <video/dw74fb.h>
#include "hx8352.h"

#ifdef CONFIG_HX8352_DEBUG
	#define PDEBUG(fmt, args...) PDEBUG( KERN_ERR "HX8352: %s(%d) " fmt, __func__, __LINE__,  ## args)
#else
	#define PDEBUG(fmt, args...) do {} while (0)
#endif 

#define LCDC_GPIO_RESET GPIO_PORTF(23)

struct dwfb_cputype_desc hx8352_cputype_desc;

static struct platform_device hx8352_pdev = {
	.name		= "hx8352",
	.dev		= {
	},
};

struct hx8352_plat_data {

#ifdef CONFIG_HAS_EARLYSUSPEND
	struct early_suspend hx8352_early_suspend;
#endif
	struct dw74fb *dw74fb;
	char* name;
};

void hx8352_disp_update(struct dw74fb *dw74fb)
{
	dw74fb_writel(dw74fb, LCDC_REG_CDISPUPR, 0x0001);
}

static void hx8352_regindex(struct dw74fb *dw74fb,unsigned short reg)
{
	// RS=0, LCMD=1 -> last command, RWCMD=1 -> WR 
	while ((dw74fb_readl(dw74fb, LCDC_REG_ISTAR1) & 0x01) == 0) {
		PDEBUG ("%s sending command\n", __func__);
	}
	dw74fb_writel(dw74fb, LCDC_REG_CMDAR, HX8352_SET_REGISTER_WRITE | (1<<6));
	udelay(200);
	dw74fb_writel(dw74fb, LCDC_REG_TXDR , reg);
	udelay(200);
}

static unsigned long hx8352_read_regvalue(struct dw74fb *dw74fb)
{
	int val;
	
	while ((dw74fb_readl(dw74fb, LCDC_REG_ISTAR1) & 0x01) == 0) {
		PDEBUG ("%s sending command\n", __func__); 
	}
	dw74fb_writel(dw74fb,LCDC_REG_CMDAR , HX8352_REGISTER_READ | (1<<6));

	val = dw74fb_readl(dw74fb,LCDC_REG_RXDR) & 0xffff;
	PDEBUG ("%s - 1st read: 0x%x\n", __func__, val);
	//readdelay (1);
	mdelay(1);
	val = dw74fb_readl(dw74fb,LCDC_REG_RXDR) & 0xffff;
	printk ("*********************************************** %s - 2nd read after delay: 0x%x\n", __func__, val);
	return val;
}

static void hx8352_write_regvalue(struct dw74fb *dw74fb,unsigned short value)
{
        /* RS=0, LCMD = NOT LAST:0 ,RWCMD:WR-1 */
	while ((dw74fb_readl(dw74fb, LCDC_REG_ISTAR1) & 0x01) == 0) {
		PDEBUG ("%s sending command\n", __func__);
	}
	dw74fb_writel(dw74fb,LCDC_REG_CMDAR,HX8352_REGISTER_WRITE | (1<<6));
	dw74fb_writel(dw74fb,LCDC_REG_TXDR, value);
}

static void hx8352_write_reg(struct dw74fb *dw74fb, int reg,int value)
{
	udelay(200);
	/* Set the index register */
	hx8352_regindex(dw74fb,reg);
	udelay(200);
	/* Send the value to be written to register */
	hx8352_write_regvalue(dw74fb,value);
	udelay(200);
}

unsigned long hx8352_read_reg(struct dw74fb *dw74fb,int reg)
{
	/* Set the index register */
	hx8352_regindex(dw74fb, reg);
	/* Send the value to be written to register */
	return hx8352_read_regvalue(dw74fb);
}

void hx8352_reset(void)
{
	//gpio_disable(GPIO_PORTC(22)); /* LCDPG3 */
	//pads_enable("lcdgp3");

	/* lcd panel reset pin */
	gpio_request(LCDC_GPIO_RESET, "hx8352_reset");
	gpio_direction_output (LCDC_GPIO_RESET, 1);
	mdelay (100);
	gpio_set_value (LCDC_GPIO_RESET, 0);
	mdelay (100);
	/* lcd normal operation */
	gpio_set_value(LCDC_GPIO_RESET, 1);
	mdelay (100);

	return;
}

#ifdef CONFIG_HAS_EARLYSUSPEND
static void hx8352_early_suspend(struct early_suspend *es) {
	lcdc_own_hw();
	/*TODO: please fill the resume sequence*/    
	lcdc_release_hw();
}

static void hx8352_late_resume(struct early_suspend *es) {
	lcdc_own_hw();
	/*TODO: please fill the resume sequence*/
	lcdc_release_hw();
}
#endif

static int hx8352_init(struct dw74fb *dw74fb)
{
	/* Reset hx8352 */
//	hx8352_reset();

//	ChipId = hx8352_read_reg(dw74fb, HX8352_PRODUCT_ID);
//	PDEBUG("******************************************************************* Error Chip Id = %x \n",ChipId);

	//if(ChipId == 0x9325) {
	//	PDEBUG(" Device Communication is success \n");
	//} else {
	//	PDEBUG("Error Chip Id = %x \n",ChipId);
	//}

	udelay (10);
	hx8352_write_reg(dw74fb,HX8352_TEST_MODE_CTRL,0x02 );                  
	hx8352_write_reg(dw74fb,HX8352_VDDD_CTRL,0x03 );                       
	hx8352_write_reg(dw74fb,HX8352_SOURCE_GAMMA_RESISTOR_SET_H,0x01 );     
	hx8352_write_reg(dw74fb,HX8352_SOURCE_GAMMA_RESISTOR_SET_L,0x93 );     
	hx8352_write_reg(dw74fb,HX8352_SYNC_FUNCTION,0x01 );                   
	hx8352_write_reg(dw74fb,HX8352_TEST_MODE_CTRL,0x00 );                  
	hx8352_write_reg(dw74fb,HX8352_GAMMA_CTRL_1,0x90 );    //0xB0 );                       
	hx8352_write_reg(dw74fb,HX8352_GAMMA_CTRL_2,0x01 );    //0x03 );                       
	hx8352_write_reg(dw74fb,HX8352_GAMMA_CTRL_3,0x10 );                    
	hx8352_write_reg(dw74fb,HX8352_GAMMA_CTRL_4,0x47 );    //0x56 );                       
	hx8352_write_reg(dw74fb,HX8352_GAMMA_CTRL_5,0x12 );    //0x13 );                       
	hx8352_write_reg(dw74fb,HX8352_GAMMA_CTRL_6,0x56 );    //0x46 );                       
	hx8352_write_reg(dw74fb,HX8352_GAMMA_CTRL_7,0x03 );    //0x23 );                       
	hx8352_write_reg(dw74fb,HX8352_GAMMA_CTRL_8,0x76 );                    
	hx8352_write_reg(dw74fb,HX8352_GAMMA_CTRL_9,0x05 );    //0x00 );                       
	hx8352_write_reg(dw74fb,HX8352_GAMMA_CTRL_10,0x57 );   //0x5E );                       
	hx8352_write_reg(dw74fb,HX8352_GAMMA_CTRL_11,0x47 );   //0x4F );                       
	hx8352_write_reg(dw74fb,HX8352_GAMMA_CTRL_12,0x45 );   //0x40 );                       
	/*Power On sequence*/
	hx8352_write_reg(dw74fb,HX8352_OSC_CTRL_1,0x91);       //0x91 );                       
	hx8352_write_reg(dw74fb,HX8352_CYCLE_CTRL_1,0xF9);     //0xF9 );                       
	hx8352_write_reg(dw74fb,HX8352_CYCLE_CTRL_4,0x48);     //0x08);                                        
	udelay(1000);
	hx8352_write_reg(dw74fb,HX8352_POWER_CTRL_3,0x14 );                    
	hx8352_write_reg(dw74fb,HX8352_POWER_CTRL_2,0x11 );                    
	hx8352_write_reg(dw74fb,HX8352_POWER_CTRL_4,0x06 );                    
	hx8352_write_reg(dw74fb,HX8352_VCOM_CTRL, 0x59 );      //0x4a );       //0x5a );                       
	udelay(200);
	hx8352_write_reg(dw74fb,HX8352_POWER_CTRL_1,0x0A );                    
	hx8352_write_reg(dw74fb,HX8352_POWER_CTRL_1,0x1A );                    
	udelay(400);
	hx8352_write_reg(dw74fb,HX8352_POWER_CTRL_1,0x12 );                    
	udelay(400);
	hx8352_write_reg(dw74fb,HX8352_POWER_CTRL_6,0x2B );    //0x2A );       //0x27 );       //0x2C );                       
	udelay(1000);
	hx8352_write_reg(dw74fb,HX8352_DISPLAY_CTRL_2,0x60 );                  
	hx8352_write_reg(dw74fb,HX8352_SOURCE_CTRL_2,0x40 );                   
	hx8352_write_reg(dw74fb,HX8352_CYCLE_CTRL_10,0x38 );                   
	hx8352_write_reg(dw74fb,HX8352_CYCLE_CTRL_11,0x38 );                   
	hx8352_write_reg(dw74fb,HX8352_DISPLAY_CTRL_2,0x38 );                  
	udelay(400);
	hx8352_write_reg(dw74fb,HX8352_DISPLAY_CTRL_2,0x3C );                  
	hx8352_write_reg(dw74fb,HX8352_MEMORY_ACCESS_CTRL,0x1C );              
	hx8352_write_reg(dw74fb,HX8352_DISPLAY_MODE_CTRL,0x06 );               
	hx8352_write_reg(dw74fb,HX8352_PANEL_CTRL,0x00 );                      
	hx8352_write_reg(dw74fb,HX8352_COL_ADDR_START_H,0x00 );                
	hx8352_write_reg(dw74fb,HX8352_COL_ADDR_START_L,0x00 );                
	hx8352_write_reg(dw74fb,HX8352_COL_ADDR_END_H,0x00 );                  
	hx8352_write_reg(dw74fb,HX8352_COL_ADDR_END_L,0xef );                  
	hx8352_write_reg(dw74fb,HX8352_ROW_ADDR_START_H,0x00 );                
	hx8352_write_reg(dw74fb,HX8352_ROW_ADDR_START_L,0x00 );                
	hx8352_write_reg(dw74fb,HX8352_ROW_ADDR_END_H,0x01 );                  
	hx8352_write_reg(dw74fb,HX8352_ROW_ADDR_END_L,0x8f );                  
	/* write the last register */
	hx8352_regindex(dw74fb,HX8352_WRITE_DATA);
	hx8352_disp_update (dw74fb);

	return 0;
}

/* associate with framebuffer */
static int hx8352_associate(struct dw74fb *dw74fb)
{
	struct hx8352_plat_data *pdata;

	pdata = kzalloc(sizeof(struct hx8352_plat_data), GFP_KERNEL);
	if (!pdata)
		return -ENOMEM;

	/* associate platform data with device */
	hx8352_cputype_desc.pdev->dev.platform_data = pdata;

	/* associate framebuffer */
	dev_set_drvdata(&dw74fb->panel->cputype_desc->pdev->dev, dw74fb);

	pdata->dw74fb = dw74fb;

	#ifdef CONFIG_HAS_EARLYSUSPEND
		pdata->hx8352_early_suspend.suspend = hx8352_early_suspend;
		pdata->hx8352_early_suspend.resume = hx8352_late_resume;
		pdata->hx8352_early_suspend.level = EARLY_SUSPEND_LEVEL_BLANK_SCREEN;
		register_early_suspend(&pdata->hx8352_early_suspend);
	#endif
	
	return 0;
}

/* init screen registers */
static int hx8352_init_regs(struct dw74fb *dw74fb)
{
	hx8352_reset();
	mdelay(100);
	hx8352_init(dw74fb);
	mdelay(100);
	hx8352_init(dw74fb);
	
	return 0;
}

/* reset screen position */
static void hx8352_prepare_screen(struct dw74fb *dw74fb)
{
	hx8352_regindex(dw74fb,HX8352_WRITE_DATA);
}

/* cputype ops */
struct dwfb_cputype_desc hx8352_cputype_desc = {
	.associate 		= hx8352_associate,
	.init_regs 		= hx8352_init_regs,
	.prepare_screen = hx8352_prepare_screen,
	.pdev 			= &hx8352_pdev,
};

static int __devinit dw_hx8352_probe(struct platform_device *pdev)
{
	return 0;
}

static int __devexit dw_hx8352_remove(struct platform_device *pdev)
{
	struct hx8352_plat_data *pdata = pdev->dev.platform_data;

	unregister_early_suspend(&pdata->hx8352_early_suspend);

	kfree(pdev->dev.platform_data);
    return 0;
}


static struct platform_driver dw_hx8352_driver = {
	.probe = dw_hx8352_probe,
	.remove = __exit_p(dw_hx8352_remove),
	.driver = {
		.name = "hx8352",
		.owner = THIS_MODULE,
	},
};

static int __init dw_hx8352_init(void)
{
    return platform_driver_register(&dw_hx8352_driver);
}

static void __exit dw_hx8352_exit(void)
{
	platform_driver_unregister(&dw_hx8352_driver);
}

module_init(dw_hx8352_init);
module_exit(dw_hx8352_exit);

MODULE_AUTHOR("DSP Group");
MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("LCD hx8352");
