/*
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#include <linux/types.h>
#include <linux/cdev.h>
#include <linux/platform_device.h>
#include <linux/gpio.h>
#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/err.h>
#include <linux/irq.h>
#include <linux/uaccess.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>

#include <asm/io.h>
#include <mach/platform.h>
#ifdef CONFIG_DP52
#include <linux/dp52.h>
#endif

#include <mach/cradledet.h>


#define DEVICE_NAME	                       "CRADLEDET"

/******************************************************************************************/
//void snd_wm8976_config_cradle_speakers( eCradle_stat  cradle_state,
//							 bool          extra_debounce_delay,
//							 uint          delay_ms );

/******************************************************************************************/


/* Cradle Detection - Putting-in or Taking-out the Handset device In/Out of the Cradle.
 *
 * On IMH/AHS boards, the indication Oscillates if charger volatge
 * is less then the 6V planned (e.g. on "basic" BDB test boards, it is only 5V).
 * IMH's R117 must be enlarged!
 * So meanwhile, Cradle Detection is Inhibited by default.
 */


/******************************************************************************************/
static struct cradledet_dev *cd;



static int dw_cradledet_read_proc(char *page, char **start, off_t off, int count, 
	int *eof, void *data){
	char *out = page;
	int len;

	out += sprintf(out, "%d\n",cd->debouncetime);
	
	len = out - page - off;
	if (len < count) {
		*eof = 1;
		if (len <= 0) return 0;
	} else {
		len = count;
	}
	*start = page + off;
	return len;

}

static int dw_cradledet_write_proc(struct file *file, const char *buf, 
	unsigned long count, void *data){

	char *lbuf[count + 1];

	if (!capable(CAP_SYS_ADMIN))
		return -EACCES;

	memset(lbuf, 0, count + 1);

	if (copy_from_user(lbuf, buf, count))
		return -EFAULT;

	sscanf((const char*)lbuf, "%d",&cd->debouncetime);

	return count;
	
}

static int __init dw_cradledet_create_procfs(void){
	struct proc_dir_entry *proc_headset_root = NULL;
	struct proc_dir_entry *ent;

	proc_headset_root = proc_mkdir("dock", 0);
	if (!proc_headset_root) return -1;
	proc_headset_root->owner = THIS_MODULE;
	ent = create_proc_entry("debouncetime", S_IFREG|S_IRUGO|S_IWUSR, proc_headset_root);
	if (!ent) return -1;
	ent->data = (void *)NULL; 
	ent->read_proc = dw_cradledet_read_proc;
	ent->write_proc = dw_cradledet_write_proc;
	ent->owner = THIS_MODULE;

	return 0;
}


static void dw_cradledet_debug_print( char *str ){
	printk(KERN_ERR"CRADLE det %s\n", str);
}

#ifndef CONFIG_CRADLEDET_POLLING
static void dw_enable_cradledet_interrupt(void){
	dp52_direct_read_modify_write(DP52_PMINTMSK, BIT_MASK(1),BIT_MASK(1));		
}

static void dw_disable_cradledet_interrupt(void){
	dp52_direct_read_modify_write(DP52_PMINTMSK, BIT_MASK(1),0);		
}
#endif


/******************************************************************************************/
static int dw_cradledet_poll_thread( void *data ){
#ifdef CONFIG_DP52
	int cradle_val1=0;
	int cradle_val2=0;
	struct cradledet_cb *pCradledetcb;
#endif

	/* Note: 
	 *
	 * HeadSet Plugged-In/Plugged-Out Detection in IMH & AHS board:
	 *    is based on "HEAD_DET" line state.
	 *    This line is connected to DP-52 HS_DET input, and read from the DP-52 register 0x44,
	 *    (RAWPMINSTAT), bit 5 (HSDETINT): '0' - Plugged Out; '1' - Plugged In.
     * 
	 * Handset On/Off Cradle Detection in IMH & AHS boards:
	 *    is based on "DCINS" line state.
	 *    This line is connected to DP-52 DCINS input, and read from DP-52 register 0x47,
	 *    (PMURAWSTAT), bit 1 (DCINS): : '0' - HS Off cradle; '1' - HS On cradle.
	 *
	 *    In IMH board only, another way is available for Cradle detection:
	 *    "Dock_DetN" line state.
	 *    This line is connected directly to AGPIO18: '0' - HS On cradle; '1' - HS Off cradle.
	 *    However, this circuit is not found in AHS board, so we don't use it.
	 */
	 
	daemonize("dw_cradledet_poll_thread");
	allow_signal(SIGTERM);

	while (1) {
#ifdef CONFIG_DP52
   
			/* Read physical Cradle state */
			cradle_val1 = dp52_direct_read(DP52_PMURAWSTAT); // DP-52 reg 0x77
			cradle_val1 = (cradle_val1 & BIT_MASK(1)) ? 1 : 0;

			/* Wait, to Debounce */
			msleep_interruptible(cd->debouncetime); // 20 mSec sleep
		
			/* Read physical Cradle state */
			cradle_val2 = dp52_direct_read(DP52_PMURAWSTAT); // DP-52 reg 0x77
			cradle_val2 = (cradle_val2 & BIT_MASK(1)) ? 1 : 0;
#else

#error "add gpio based detection code here"

#endif

			if (cradle_val2 != cradle_val1) {
				/* Restore former steady-state */
				cradle_val2 = cd->cradle_state;
			}else if (cradle_val2 != cd->cradle_state) {

					cd->cradle_state = cradle_val2;
					
					/* print current detection status */
					if (cd->cradle_state == 1) {
						dw_cradledet_debug_print("dw_cradledet_poll_thread - on cradle state" );
					} else {
						dw_cradledet_debug_print("dw_cradledet_poll_thread - off cradle state" );
					}
				    
					/* Update the sysfs headset state with that of headset */
                 	switch_set_state(&cd->cradledet_sdev, cd->cradle_state);
					
                    spin_lock(&cd->pCBfnQueue_lock);
					list_for_each_entry(pCradledetcb, &cd->pCBfnQueue, list) {
						/* Call the Client's Callback function */
						pCradledetcb->pCradleDetCBfn(cd->cradle_state);

					}
                    spin_unlock(&cd->pCBfnQueue_lock);

#ifndef CONFIG_CRADLEDET_POLLING /* if we are in interrupt mode */

					if (cd->cradle_state == 0) {
						/* we are off cradle then enable the interrupt and exit the thread */
						dw_cradledet_debug_print("dw_cradledet_poll_thread - off cradle state, enabling interrupt and exiting");
						/* Enable the interrupt */
						dw_enable_cradledet_interrupt();
	
						return 0;
					}
#endif
			}
		msleep_interruptible(cd->poll_period); 
	}
   
	return 0;
} // end dw_cradledet_poll_thread()



bool dw_cradledet_cb_register(void (*pfn)(int)){
    struct cradledet_cb *pCradledetcb;

	pCradledetcb = kzalloc(sizeof(struct cradledet_cb),GFP_KERNEL);
	if(!pCradledetcb) return 0;

	pCradledetcb->pCradleDetCBfn = pfn;

	spin_lock(&cd->pCBfnQueue_lock);
	list_add_tail(&pCradledetcb->list,&cd->pCBfnQueue);
	spin_unlock(&cd->pCBfnQueue_lock);

	return 1;
}

EXPORT_SYMBOL(dw_cradledet_cb_register);



bool dw_cradledet_cb_unregister(void (*pfn)(int)){
    struct cradledet_cb *pCradledetcb;
	
    spin_lock(&cd->pCBfnQueue_lock);
	list_for_each_entry(pCradledetcb, &cd->pCBfnQueue, list) {
		if(pCradledetcb->pCradleDetCBfn == pfn)
		{
			list_del(&pCradledetcb->list);
		    kfree(pCradledetcb);
			break;
		}
	}
    spin_unlock(&cd->pCBfnQueue_lock);			
	
	return 1;
}

EXPORT_SYMBOL(dw_cradledet_cb_unregister);

unsigned int dw_cradledet_get_capability(void){
	return cd->capability;
}

EXPORT_SYMBOL(dw_cradledet_get_capability);


int dw_cradledet_get_cradlestate(void){
    int	cradle_state;

	spin_lock(&cd->cradle_state_lock);
	cradle_state = cd->cradle_state;
	spin_unlock(&cd->cradle_state_lock);

	return cradle_state;
}

EXPORT_SYMBOL(dw_cradledet_get_cradlestate);


static void dw_cradledet_work(struct work_struct *work){

	int cradle_val1,cradle_val2;
	struct cradledet_cb *pCradledetcb;

	dw_cradledet_debug_print("dw_cradledet_work");

#ifdef CONFIG_DP52
	/* Read physical Cradle state */
	cradle_val1 = dp52_direct_read(DP52_PMURAWSTAT); // DP-52 reg 0x77
	cradle_val1 = (cradle_val1 & BIT_MASK(1)) ? 1 : 0;

	/* Wait, to Debounce */
	msleep_interruptible(cd->debouncetime); // 20 mSec sleep

	/* Read physical Cradle state */
	cradle_val2 = dp52_direct_read(DP52_PMURAWSTAT); // DP-52 reg 0x77
	cradle_val2 = (cradle_val2 & BIT_MASK(1)) ? 1 : 0;
#else
	#error " complete here gpio based detection"

#endif
	if (cradle_val2 != cradle_val1) {
			/* Restore former steady-state */
			cradle_val2 = cd->cradle_state;

	}else if (cradle_val2 != cd->cradle_state) {

		dw_cradledet_debug_print("dw_cradledet_work - on cradle state");

			cd->cradle_state = cradle_val2;

			switch_set_state(&cd->cradledet_sdev, cd->cradle_state);

			spin_lock(&cd->pCBfnQueue_lock);
			list_for_each_entry(pCradledetcb, &cd->pCBfnQueue, list) {
				/* Call the Client's Callback function */
				pCradledetcb->pCradleDetCBfn(cd->cradle_state);
			}
			spin_unlock(&cd->pCBfnQueue_lock);

			/* now lunch the thread */
			dw_cradledet_debug_print("dw_cradledet_work - on cradle lunching thread");

			kernel_thread(dw_cradledet_poll_thread, NULL, CLONE_KERNEL);
	}
}

#ifndef CONFIG_CRADLEDET_POLLING

#ifdef CONFIG_DP52

static irqreturn_t dw_cradledet_event_handler(void *dev, int event){

    dp52_direct_read_modify_write(DP52_PMUSTAT, DP52_PMUSTAT_DDCINS, 0x0);
 
    dw_disable_cradledet_interrupt();
#if 0
	cd->debouncetimer.expires = jiffies + cd->debouncetime;    /* timer expires in delay ticks */
	cd->debouncetimer.data = 0;                                         /* zero is passed to the timer handler */
	cd->debouncetimer.function = dw_cradledet_timer_debounce_cb;  

    add_timer(&cd->debouncetimer);
#endif
	// call work Q for doing the de-bounce
	queue_work(cd->cradledet_wq, &cd->cradledet_work);

	return IRQ_HANDLED;
}

#else

static int dw_cradledet_interrupt(int irq, void *id){
	
	/* add disable interrut code here ? */
	queue_work(cd->cradledet_wq, &cd->cradledet_work);

	return IRQ_HANDLED;
}

#endif /* CONFIG_DP52 */

#endif /* CONFIG_CRADLEDET_POLLING */

static void dw_enable_cradledet( struct platform_device *pdev  ){
	eCradle_stat  cradle_stat;
	struct dwcradledet_config *pdwcradledet_config;
#ifdef CONFIG_DP52
	int 		  val;
#endif

	dw_cradledet_debug_print("dw_enable_cradledet");

#if 0 /* cradle gpio handling is in audio driver resposability */
	/* Allocate DGPIO23 gpio for Cradle speakers' Power Amp MUTE */
	cd->dock_mute_gpio = GPIO_PORTD(23);
	if (gpio_request(cd->dock_mute_gpio, DEVICE_NAME)) {
		printk(KERN_ERR"dw_enable_cradledet(): Allocating DGPIO23 on IMH board Failed!\n");
	}
	gpio_set_enable(cd->dock_mute_gpio, 1);
	/* Note: Setting this gpio to be output, is done in snd_wm8976_config_cradle_speakers() */
	
	/* Allocate CGPIO0 gpio for Cradle speakers' Power Amp ~ScdN */
	cd->pa_shutn_gpio = GPIO_PORTC(0);
	if (gpio_request(cd->pa_shutn_gpio, DEVICE_NAME)) {
		printk(KERN_ERR "dw_enable_cradledet(): Allocating CGPIO0 on IMH board Failed!\n");
	}
	gpio_set_enable(cd->pa_shutn_gpio, 1);
	/* Note: Setting this gpio to be output, is done in snd_wm8976_config_cradle_speakers() */
	/* Init Cradle State according to current physical state of On/Off Cradle */
#endif 

#ifdef CONFIG_DP52
	/* Enable DP-52 CRADLE DET INTEN */
	dp52_direct_read_modify_write(DP52_PMINTMSK, BIT_MASK(1),BIT_MASK(1));
	
	val = dp52_direct_read(DP52_PMURAWSTAT); // DP-52 reg 0x77
	cd->cradle_state = (val & BIT_MASK(1)) ? 1 : 0;

	printk(KERN_ERR"cradle state = =%d\n",cd->cradle_state);

#else 
	cd->cradle_state = 0;
#endif 
	
	/* Initially configure cradle-related AFE */
	cradle_stat = (cd->cradle_state == 0) ? kCradle_stat_OFF : kCradle_stat_ON;
//	snd_wm8976_config_cradle_speakers(cradle_stat,false,0);
	
	pdwcradledet_config = (struct dwcradledet_config*)pdev->dev.platform_data;
	cd->capability = pdwcradledet_config->capability;
		
#ifdef CONFIG_CRADLEDET_POLLING		
	/* Launch AFE Detection thread */
	kernel_thread(dw_cradledet_poll_thread, NULL, CLONE_KERNEL);

#else /* if we are in interrupt driven mode */

	/*register to the on cradle interrupt */
#ifdef CONFIG_DP52
	dp52_register_irqhandler(DP52_CRADLE_DET, dw_cradledet_event_handler,pdev);
#else
	err = request_irq(pdev->dev.platform_data->irq, &dw_cradledet_interrupt, IRQF_DISABLED,
			  "CRADLE-DET", cd);
#endif

	/* first check the state of the cradle, if we are on then luanch thread else enable int */
	if (cd->cradle_state == 1){
		kernel_thread(dw_cradledet_poll_thread, NULL, CLONE_KERNEL);
	} 

#endif /* CONFIG_CRADLEDET_POLLING */

}

static ssize_t dw_cradledet_switch_print_state(struct switch_dev *sdev, char *buf){
	const char *state;
	if (switch_get_state(sdev))
		state = cd->state_on;
	else
		state = cd->state_off;

	if (state)
		return sprintf(buf, "%s\n", state);
	return -1;
}


static int __devinit dw_cradledet_probe( struct platform_device *pdev ){

	int err = 0,rc=0;
	struct dwcradledet_config *pdwcradledet_config;
	struct gpio_switch_platform_data *pswitchdata;


	dw_cradledet_debug_print("dw_cradledet_probe");

	cd = kzalloc(sizeof(struct cradledet_dev), GFP_KERNEL);

	if(cd == NULL){
		return -1;
	}
	memset(cd,sizeof(struct cradledet_dev),0);


    cd->poll_period = CRADLEDET_POLL_PERIOD;
	cd->debouncetime = CRADLEDET_DEBOUNCE_PERIOD;

	INIT_LIST_HEAD(&cd->pCBfnQueue);
	spin_lock_init(&cd->pCBfnQueue_lock);


    spin_lock_init(&cd->cradle_state_lock);

	sema_init(&(cd->poll_sem), 0);


	init_timer(&cd->debouncetimer);

	cd->cradledet_wq = create_workqueue("CradleDETWorkQueue");
	INIT_WORK(&cd->cradledet_work, dw_cradledet_work);

	/* Trigger Cradle detection thread */
	dw_enable_cradledet(pdev);
	
	pdwcradledet_config = (struct dwcradledet_config*)pdev->dev.platform_data;

	dw_cradledet_create_procfs();

	pswitchdata = &pdwcradledet_config->switch_platform_data;
	cd->cradledet_sdev.name = pswitchdata->name;
	cd->name_on = pswitchdata->name_on;
	cd->name_off = pswitchdata->name_off;
	cd->state_on = pswitchdata->state_on;
	cd->state_off = pswitchdata->state_off;
	cd->cradledet_sdev.print_state = dw_cradledet_switch_print_state;

	rc = switch_dev_register(&cd->cradledet_sdev);
	if (rc < 0){
		printk(KERN_ERR"Headset switch registration failed\n");
		return rc;
	}

	/* Initialize the sysfs headset state with that of headset */
	switch_set_state(&cd->cradledet_sdev, cd->cradle_state);

	return err;
}


static int __devexit dw_cradledet_remove( struct platform_device *devptr ){
	struct cradledet_cb *pCradledetcb;

	dw_cradledet_debug_print("dw_cradledet_remove");
	
    spin_lock(&cd->pCBfnQueue_lock);
	list_for_each_entry(pCradledetcb, &cd->pCBfnQueue, list) {
 		list_del(&pCradledetcb->list);
		kfree(pCradledetcb);
	}
    spin_unlock(&cd->pCBfnQueue_lock);

    switch_dev_unregister(&cd->cradledet_sdev);
	destroy_workqueue(cd->cradledet_wq);
	
    if(cd !=NULL)
    	kfree(cd);

	return 0;
}


static struct platform_driver dw_cradledet_driver =
{
	.probe		= dw_cradledet_probe,
	.remove		= __devexit_p(dw_cradledet_remove),
    .driver		=
	{
		.name	= "dw-cradledet",
        .owner	= THIS_MODULE,
	},
};


/**************************************************************************************************/
/* Module's Init & Exit */
/* -------------------- */


/* Init function */
static int __init dw_cradledet_init( void ){

   	return platform_driver_register(&dw_cradledet_driver);
}

/* Exit function */
static void __exit dw_cradledet_exit( void ){
   return platform_driver_unregister(&dw_cradledet_driver);
}

module_init(dw_cradledet_init);
module_exit(dw_cradledet_exit);

MODULE_AUTHOR("Murali T.D. Mohan");
MODULE_DESCRIPTION("DW74 Cradle Detection Driver");
MODULE_LICENSE("GPL");
/**************************************************************************************************/
