/*
 * drivers/rtc/rtc-dp52.c - driver for the RTC block of the DP52 chip
 *
 * Copyright (C) 2011 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <linux/delay.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/rtc.h>
#include <linux/slab.h>
#include <linux/time.h>

#include <linux/mfd/dp52/core.h>
#include <linux/rtc-dp52.h>

static char dp52_rtc_name[] = "dp52-rtc";

#define DP52_RTC_MAXVAL 0x0fffffff

#define RTC_VALID       (1 << 0)

#define SAMPLE          (1 << 15)
#define ALRMSTAT        (1 << 13)
#define ALRMINT_EN      (1 << 12)
#define ALRMON_EN       (1 << 11)

struct dp52_rtc {
	struct dp52 *dp52;
	struct rtc_device *dev;
	int irq;

	/*
	 * Those two variables build the foundation of this driver. Since the
	 * rtc only features a counter which cannot be reset and which value is
	 * unknown at system boot, we need a fixed point in time and a
	 * corresponding rtc counter sample to be able to calculate the current
	 * time. This pair is set via the settime call.
	 *
	 * rtc_reference: the rtc counter sample (one tick is one second)
	 * rtc_time:      the time at the rtc counter sample in seconds (mktime)
	 */
	unsigned long rtc_reference;
	unsigned long rtc_time;
};

static unsigned long rtc_time_to_sec(struct rtc_time *tm)
{
	return mktime(tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday,
	              tm->tm_hour, tm->tm_min, tm->tm_sec);
}

static unsigned long dp52_rtc_read_sample(struct dp52_rtc *rtc)
{
	struct dp52 *dp52 = rtc->dp52;
	unsigned long tmp, sec;

	tmp = dp52_read(dp52, DP52_RTC_5);
	dp52_write(dp52, DP52_RTC_5, tmp | SAMPLE);

	sec  = dp52_read(dp52, DP52_RTC_1) >> 4;
	sec |= dp52_read(dp52, DP52_RTC_2) << 12;

	dp52_write(dp52, DP52_RTC_5, tmp);

	return sec;
}

static irqreturn_t dp52_rtc_irq(int irq, void *priv)
{
	struct dp52_rtc *rtc = priv;

	/* clear interrupt */
	dp52_write(rtc->dp52, DP52_RTC_5, 0);

	/* update status to rtc layer */
	rtc_update_irq(rtc->dev, 1, RTC_AF);

	return IRQ_HANDLED;
}

static int dp52_rtc_readtime(struct device *dev, struct rtc_time *tm);

static int dp52_rtc_ioctl(struct device *dev, unsigned int cmd,
                          unsigned long arg)
{
	struct dp52_rtc *rtc = dev_get_drvdata(dev);
	int ret = 0;

	void __user *argp = (void __user *)arg;

	switch (cmd) {
	case RTC_AIE_ON:
		break;
	case RTC_AIE_OFF:
		break;
	case RTC_DP52_REFTIME_SET:
	{
		struct rtc_dp52_reftime reftime;

		if (copy_from_user(&reftime, argp, sizeof(reftime)))
			return -EFAULT;

		rtc->rtc_time = rtc_time_to_sec(&reftime.rtc_tm);
		rtc->rtc_reference = reftime.reference;

		break;
	}
	case RTC_DP52_REFTIME_GET:
	{
		struct rtc_dp52_reftime reftime;

		reftime.reference = dp52_rtc_read_sample(rtc);
		dp52_rtc_readtime(dev, &reftime.rtc_tm);

		if (copy_to_user(argp, &reftime, sizeof(reftime)))
			return -EFAULT;

		break;
	}

	default:
		ret = -ENOIOCTLCMD;
		break;
	}

	return ret;
}

static int dp52_rtc_readtime(struct device *dev, struct rtc_time *tm)
{
	struct dp52_rtc *rtc = dev_get_drvdata(dev);
	unsigned long sample;

	/*
	 * get the sample first, this will also retrieve the valid bit
	 * used below
	 */
	sample = dp52_rtc_read_sample(rtc);

	/*
	 * our time is only valid if we have a valid point in time and the
	 * rtc was continuously powered since the last settime()
	 */
	if (!rtc->rtc_time || !(dp52_read(rtc->dp52, DP52_RTC_1) & RTC_VALID))
		return -1;

	/*
	 * handle wrap around. we simply add the DP52_RTC_MAXVAL to sample
	 * since it is far less what a 32bit value can store.
	 */
	if (sample < rtc->rtc_reference)
		sample += DP52_RTC_MAXVAL;

	rtc_time_to_tm(rtc->rtc_time + sample - rtc->rtc_reference, tm);

	return 0;
}

static int dp52_rtc_settime(struct device *dev, struct rtc_time *tm)
{
	struct dp52_rtc *rtc = dev_get_drvdata(dev);

	rtc->rtc_reference = dp52_rtc_read_sample(rtc);
	rtc->rtc_time = mktime(tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday,
	                       tm->tm_hour, tm->tm_min, tm->tm_sec);

	dp52_write(rtc->dp52, DP52_RTC_1, RTC_VALID);

	return 0;
}

static int dp52_rtc_readalarm(struct device *dev, struct rtc_wkalrm *wkalrm)
{
	struct dp52_rtc *rtc = dev_get_drvdata(dev);
	struct dp52 *dp52 = rtc->dp52;
	unsigned long alarm;
	struct rtc_time *tm = &wkalrm->time;

	alarm  = dp52_read(dp52, DP52_RTC_4) >> 4;
	alarm |= dp52_read(dp52, DP52_RTC_3) << 12;

	wkalrm->enabled = !!(dp52_read(dp52, DP52_RTC_5) & ALRMON_EN);

	if (alarm < rtc->rtc_reference)
		alarm += DP52_RTC_MAXVAL;

	alarm = alarm - rtc->rtc_reference + rtc->rtc_time;

	rtc_time_to_tm(alarm, tm);

	return 0;
}

static int dp52_rtc_setalarm(struct device *dev, struct rtc_wkalrm *wkalrm)
{
	struct dp52_rtc *rtc = dev_get_drvdata(dev);
	unsigned long alarm;
	struct rtc_time *tm = &wkalrm->time;

	alarm = mktime(tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday,
	               tm->tm_hour, tm->tm_min, tm->tm_sec);
	alarm = alarm - rtc->rtc_time + rtc->rtc_reference;

	if (alarm > DP52_RTC_MAXVAL)
		alarm -= DP52_RTC_MAXVAL;

	dp52_write(rtc->dp52, DP52_RTC_3, (alarm & 0x0ffff000) >> 12);
	dp52_write(rtc->dp52, DP52_RTC_4, (alarm & 0x00000fff) << 4);
	dp52_write(rtc->dp52, DP52_RTC_5, ALRMON_EN | ALRMINT_EN);

	return 0;
}

static int dp52_rtc_proc(struct device *dev, struct seq_file *seq)
{
	return 0;
}

static const struct rtc_class_ops dp52_rtc_ops = {
	.ioctl = dp52_rtc_ioctl,
	.read_time = dp52_rtc_readtime,
	.set_time = dp52_rtc_settime,
	.read_alarm = dp52_rtc_readalarm,
	.set_alarm = dp52_rtc_setalarm,
	.proc = dp52_rtc_proc,
};

static int __devinit dp52_rtc_probe(struct platform_device *pdev)
{
	struct dp52 *dp52 = dev_get_drvdata(pdev->dev.parent);
	struct dp52_rtc *rtc;
	int ret;

	rtc = kzalloc(sizeof(*rtc), GFP_KERNEL);
	if (!rtc)
		return -ENOMEM;

	platform_set_drvdata(pdev, rtc);

	rtc->dp52 = dp52;
	if ((rtc->irq = platform_get_irq(pdev, 0)) < 0) {
		ret = rtc->irq;
		dev_err(&pdev->dev, "no IRQ!\n");
		goto err_free;
	}

	rtc->dev = rtc_device_register(dp52_rtc_name, &pdev->dev, &dp52_rtc_ops,
	                               THIS_MODULE);
	if (IS_ERR(rtc->dev)) {
		ret = PTR_ERR(rtc->dev);
		dev_err(&pdev->dev, "cannot allocate RTC: %d\n", ret);
		goto err_free;
	}

	/* reset alarm */
	dp52_write(dp52, DP52_RTC_5, 0);
	dp52_write(dp52, DP52_RTC_3, 0);
	dp52_write(dp52, DP52_RTC_4, 0);

	ret = request_threaded_irq(rtc->irq, NULL, dp52_rtc_irq, 0,
	                           dp52_rtc_name, rtc);
	if (ret) {
		dev_err(&pdev->dev, "failed to request IRQ %d: %d\n", rtc->irq, ret);
		goto err_unregister;
	}

	device_init_wakeup(&pdev->dev, 1);

	dev_info(&pdev->dev, "initialized\n");
	return 0;

err_unregister:
	rtc_device_unregister(rtc->dev);
err_free:
	kfree(rtc);
	return ret;
}

static int __exit dp52_rtc_remove(struct platform_device *pdev)
{
	struct dp52_rtc *rtc = platform_get_drvdata(pdev);

	free_irq(rtc->irq, rtc);
	rtc_device_unregister(rtc->dev);
	kfree(rtc);

	return 0;
}

#ifdef CONFIG_PM
static int dp52_rtc_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct dp52_rtc *rtc = platform_get_drvdata(pdev);

	if (device_may_wakeup(&pdev->dev))
		enable_irq_wake(rtc->irq);

	return 0;
}

static int dp52_rtc_resume(struct platform_device *pdev)
{
	struct dp52_rtc *rtc = platform_get_drvdata(pdev);

	if (device_may_wakeup(&pdev->dev))
		disable_irq_wake(rtc->irq);

	return 0;
}
#else
#define dp52_rtc_suspend NULL
#define dp52_rtc_resume NULL
#endif

static struct platform_driver dp52_rtc_driver = {
	.remove  = __exit_p(dp52_rtc_remove),
	.probe	 = dp52_rtc_probe,
	.suspend = dp52_rtc_suspend,
	.resume  = dp52_rtc_resume,
	.driver  = {
		.name = dp52_rtc_name,
		.owner = THIS_MODULE,
	},
};

static int __init dp52_rtc_init(void)
{
	return platform_driver_register(&dp52_rtc_driver);
}

static void __exit dp52_rtc_exit(void)
{
	platform_driver_unregister(&dp52_rtc_driver);
}

module_init(dp52_rtc_init);
module_exit(dp52_rtc_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("DSPG Technologies GmbH");
MODULE_DESCRIPTION("Driver for the RTC block of the DSPG DP52");

