/******************************************************************************
 *{@C
 *      Copyright:      2009-2022 Paul Obermeier (obermeier@tcl3d.org)
 *
 *                      See the file "Tcl3D_License.txt" for information on
 *                      usage and redistribution of this file, and for a
 *                      DISCLAIMER OF ALL WARRANTIES.
 *
 *      Module:         Tcl3D -> tcl3dOsg
 *      Filename:       tcl3dMixinVector.i
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    SWIG file for wrapping the Open Scene Graph library.
 *                      This file is a modified version of the Swig file
 *                      for wrapping std::vector (std_vector.i) for Tcl.
 *
 *****************************************************************************/

/* -----------------------------------------------------------------------------
 * See the LICENSE file for information on copyright, usage and redistribution
 * of SWIG, and the README file for authors - http://www.swig.org/release.html.
 *
 * tcl3dMixinVector.i
 * ----------------------------------------------------------------------------- */

%include <std_common.i>

// ------------------------------------------------------------------------
// osg::MixinVector
// 
// The aim of all that follows would be to integrate osg::MixinVector with 
// Tcl as much as possible, namely, to allow the user to pass and 
// be returned Tcl lists.
// const declarations are used to guess the intent of the function being
// exported; therefore, the following rationale is applied:
// 
//   -- f(osg::MixinVector<T>), f(const osg::MixinVector<T>&), f(const osg::MixinVector<T>*):
//      the parameter being read-only, either a Tcl list or a
//      previously wrapped osg::MixinVector<T> can be passed.
//   -- f(osg::MixinVector<T>&), f(osg::MixinVector<T>*):
//      the parameter must be modified; therefore, only a wrapped osg::MixinVector
//      can be passed.
//   -- osg::MixinVector<T> f():
//      the vector is returned by copy; therefore, a Tcl list of T:s 
//      is returned which is most easily used in other Tcl functions procs
//   -- osg::MixinVector<T>& f(), osg::MixinVector<T>* f(), const osg::MixinVector<T>& f(),
//      const osg::MixinVector<T>* f():
//      the vector is returned by reference; therefore, a wrapped osg::MixinVector
//      is returned
// ------------------------------------------------------------------------

%{
#include <vector>
#include <algorithm>
#include <stdexcept>
#include <string>

Tcl_Obj* SwigMixinString_FromString(const std::string &s) {
    return Tcl_NewStringObj(s.data(), (int)s.length());
}

int SwigMixin_GetBoolFromObj(Tcl_Interp *interp, Tcl_Obj *o, bool *val) {
  int v;
  int res = Tcl_GetBooleanFromObj(interp, o, &v);
  if (res == TCL_OK) {
    *val = v ? true : false;
  }
  return res;  
}
 
int SwigMixinString_AsString(Tcl_Interp *interp, Tcl_Obj *o, std::string *val) {
    int len;
    const char* temp = Tcl_GetStringFromObj(o, &len);
    if (temp == NULL)
        return TCL_ERROR;
    val->assign(temp, len);
    return TCL_OK;
}

// behaviour of this is such as the real Tcl_GetIntFromObj
template <typename Type>
int SwigMixinInt_As(Tcl_Interp *interp, Tcl_Obj *o, Type *val) {
    int temp_val, return_val;
    return_val = Tcl_GetIntFromObj(interp, o, &temp_val);
    *val = (Type) temp_val;
    return return_val;
}

// behaviour of this is such as the real Tcl_GetDoubleFromObj
template <typename Type>
int SwigMixinDouble_As(Tcl_Interp *interp, Tcl_Obj *o, Type *val) {
    int return_val;
    double temp_val;
    return_val = Tcl_GetDoubleFromObj(interp, o, &temp_val);
    *val = (Type) temp_val;
    return return_val;
}

%}

// exported class

namespace osg {
    
    template<class T> class MixinVector {
        %typemap(in) MixinVector<T> (osg::MixinVector<T> *v) {
            Tcl_Obj **listobjv;
            int       nitems;
            int       i;
            T*        temp;

            if (SWIG_ConvertPtr($input, (void **) &v, \
                                $&1_descriptor, 0) == 0){
                $1 = *v;
            } else {
                // Case 1
                // It isn't a MixinVector<T> so it should be a list of T's
                if(Tcl_ListObjGetElements(interp, $input, \
                                          &nitems, &listobjv) == TCL_ERROR)
                    return TCL_ERROR;
                $1 = osg::MixinVector<T>();
                for (i = 0; i < nitems; i++) {
                    if ((SWIG_ConvertPtr(listobjv[i],(void **) &temp,
                                         $descriptor(T *),0)) != 0) {
                        char message[] = 
                            "list of " #T " expected";
                        Tcl_SetResult(interp, message, TCL_VOLATILE);
                        return TCL_ERROR;
                    }
                    $1.push_back(*temp);
                } 
            }
        }

        %typemap(in) const MixinVector<T>* (osg::MixinVector<T> *v, osg::MixinVector<T> w),
                     const MixinVector<T>& (osg::MixinVector<T> *v, osg::MixinVector<T> w) {
            Tcl_Obj **listobjv;
            int       nitems;
            int       i;
            T*        temp;

            if(SWIG_ConvertPtr($input, (void **) &v, \
                               $1_descriptor, 0) == 0) {
                $1 = v;
            } else {
                // Case 2
                // It isn't a MixinVector<T> so it should be a list of T's
                if(Tcl_ListObjGetElements(interp, $input, 
                                          &nitems, &listobjv) == TCL_ERROR)
                    return TCL_ERROR;
                w = osg::MixinVector<T>();
                for (i = 0; i < nitems; i++) {
                    if ((SWIG_ConvertPtr(listobjv[i],(void **) &temp,
                                         $descriptor(T *),0)) != 0) {
                        char message[] = 
                            "list of " #T " expected";
                        Tcl_SetResult(interp, message, TCL_VOLATILE);
                        return TCL_ERROR;
                    }
                    w.push_back(*temp);
                } 
                $1 = &w;
            }
        }

        %typemap(out) MixinVector<T> {
            for (unsigned int i=0; i<$1.size(); i++) {
                T* ptr = new T((($1_type &)$1)[i]);
                Tcl_ListObjAppendElement(interp, $result, \
                                         SWIG_NewInstanceObj(ptr, 
                                                             $descriptor(T *), 
                                                             0));
            }
        }

        %typecheck(SWIG_TYPECHECK_VECTOR) MixinVector<T> {
            Tcl_Obj **listobjv;
            int       nitems;
            T*        temp;
            osg::MixinVector<T> *v;
            
            if(SWIG_ConvertPtr($input, (void **) &v, \
                               $&1_descriptor, 0) == 0) {
                /* wrapped MixinVector */
                $1 = 1;
            } else {
                // Case 3
                // It isn't a MixinVector<T> so it should be a list of T's
                if(Tcl_ListObjGetElements(interp, $input, 
                                          &nitems, &listobjv) == TCL_ERROR)
                    $1 = 0;
                else
                    if (nitems == 0)
                        $1 = 1;
                //check the first value to see if it is of correct type
                    else if ((SWIG_ConvertPtr(listobjv[0],
                                              (void **) &temp, 
                                              $descriptor(T *),0)) != 0)
                        $1 = 0;
                    else
                        $1 = 1;
            }
        }
        
        %typecheck(SWIG_TYPECHECK_VECTOR) const MixinVector<T>&,
                                          const MixinVector<T>* {
            Tcl_Obj **listobjv;
            int       nitems;
            T*         temp;
            osg::MixinVector<T> *v;

            if(SWIG_ConvertPtr($input, (void **) &v, \
                               $1_descriptor, 0) == 0){
                /* wrapped MixinVector */
                $1 = 1;
            } else {
                // Case 4
                // It isn't a MixinVector<T> so it should be a list of T's
                if(Tcl_ListObjGetElements(interp, $input, 
                                          &nitems, &listobjv) == TCL_ERROR)
                    $1 = 0;
                else
                    if (nitems == 0)
                        $1 = 1;
                //check the first value to see if it is of correct type
                    else if ((SWIG_ConvertPtr(listobjv[0],
                                              (void **) &temp,
                                              $descriptor(T *),0)) != 0)
                        $1 = 0;
                    else
                        $1 = 1;
            }
        }
      
      public:
        MixinVector(unsigned int size = 0);
        MixinVector(unsigned int size, const T& value);
        MixinVector(const MixinVector<T> &);

#if OPA_TODO
        unsigned int size() const;
        bool empty() const;
#else
        unsigned int size();
        bool empty();
#endif
        void clear();
        %rename(push) push_back;
        void push_back(const T& x);
        %extend {
            T pop() throw (std::out_of_range) {
                if (self->size() == 0)
                    throw std::out_of_range("pop from empty MixinVector");
                T x = self->back();
                self->pop_back();
                return x;
            }
            T get(int i) throw (std::out_of_range) {
                int size = int(self->size());
                if (i<0) i += size;
                if (i>=0 && i<size)
                    return (*self)[i];
                else
                    throw std::out_of_range("MixinVector index out of range");
            }
            void set(int i, const T& x) throw (std::out_of_range) {
                int size = int(self->size());
                if (i<0) i+= size;
                if (i>=0 && i<size)
                    (*self)[i] = x;
                else
                    throw std::out_of_range("MixinVector index out of range");
            }
        }
    };


    // specializations for built-ins

    %define specialize_osg_MixinVector(T, CONVERT_FROM, CONVERT_TO)
    template<> class MixinVector<T> {

        %typemap(in) MixinVector<T> (osg::MixinVector<T> *v){
            Tcl_Obj **listobjv;
            int       nitems;
            int       i;
            T         temp;

            if(SWIG_ConvertPtr($input, (void **) &v, \
                               $&1_descriptor, 0) == 0) {
                $1 = *v;
            } else {
                // It isn't a MixinVector<T> so it should be a list of T's
                if(Tcl_ListObjGetElements(interp, $input, 
                                          &nitems, &listobjv) == TCL_ERROR)
                    return TCL_ERROR;					      
                $1 = osg::MixinVector<T>();
                for (i = 0; i < nitems; i++) {
                    if (CONVERT_FROM(interp, listobjv[i], &temp) == TCL_ERROR)
                        return TCL_ERROR;
                    $1.push_back(temp);
                } 
            }
        }
      
        %typemap(in) const MixinVector<T>& (osg::MixinVector<T> *v,osg::MixinVector<T> w),
                     const MixinVector<T>* (osg::MixinVector<T> *v,osg::MixinVector<T> w) {
            Tcl_Obj **listobjv;
            int       nitems;
            int       i;
            T         temp;

            if(SWIG_ConvertPtr($input, (void **) &v, \
                               $1_descriptor, 0) == 0) {
                $1 = v;
            } else {
                // It isn't a MixinVector<T> so it should be a list of T's
                if(Tcl_ListObjGetElements(interp, $input, 
                                          &nitems, &listobjv) == TCL_ERROR)
                    return TCL_ERROR;
                w = osg::MixinVector<T>();
                for (i = 0; i < nitems; i++) {
                    if (CONVERT_FROM(interp, listobjv[i], &temp) == TCL_ERROR)
                        return TCL_ERROR;
                    w.push_back(temp);
                } 
                $1 = &w;
            }
        }

        %typemap(out) MixinVector<T> {
            for (unsigned int i=0; i<$1.size(); i++) {
                Tcl_ListObjAppendElement(interp, $result, \
                                         CONVERT_TO((($1_type &)$1)[i]));
            }
        }
       
        %typecheck(SWIG_TYPECHECK_VECTOR) MixinVector<T> {
            Tcl_Obj **listobjv;
            int       nitems;
            T         temp;
            osg::MixinVector<T> *v;

            if(SWIG_ConvertPtr($input, (void **) &v, \
                               $&1_descriptor, 0) == 0){
                /* wrapped MixinVector */
                $1 = 1;
            } else {
                // It isn't a MixinVector<T> so it should be a list of T's
                if(Tcl_ListObjGetElements(interp, $input, 
                                          &nitems, &listobjv) == TCL_ERROR)
                    $1 = 0;
                else
                    if (nitems == 0)
                        $1 = 1;
                //check the first value to see if it is of correct type
                if (CONVERT_FROM(interp, listobjv[0], &temp) == TCL_ERROR)
                    $1 = 0;
                else
                    $1 = 1;
            }
        }      

        %typecheck(SWIG_TYPECHECK_VECTOR) const MixinVector<T>&,
	                                      const MixinVector<T>*{
            Tcl_Obj **listobjv;
            int       nitems;
            T         temp;
            osg::MixinVector<T> *v;

            if(SWIG_ConvertPtr($input, (void **) &v, \
                               $1_descriptor, 0) == 0){
                /* wrapped MixinVector */
                $1 = 1;
            } else {
                // It isn't a MixinVector<T> so it should be a list of T's
                if(Tcl_ListObjGetElements(interp, $input, 
                                          &nitems, &listobjv) == TCL_ERROR)
                    $1 = 0;
                else
                    if (nitems == 0)
                        $1 = 1;
                //check the first value to see if it is of correct type
                if (CONVERT_FROM(interp, listobjv[0], &temp) == TCL_ERROR)
                    $1 = 0;
                else
                    $1 = 1;
            }
        }
        
      public:
        MixinVector(unsigned int size = 0);
        MixinVector(unsigned int size, const T& value);
        MixinVector(const MixinVector<T> &);

#if OPA_TODO
        unsigned int size() const;
        bool empty() const;
#else
        unsigned int size();
        bool empty();
#endif
        void clear();
        %rename(push) push_back;
        void push_back(T x);
        %extend {
            T pop() throw (std::out_of_range) {
                if (self->size() == 0)
                    throw std::out_of_range("pop from empty MixinVector");
                T x = self->back();
                self->pop_back();
                return x;
            }
            T get(int i) throw (std::out_of_range) {
                int size = int(self->size());
                if (i<0) i += size;
                if (i>=0 && i<size)
                    return (*self)[i];
                else
                    throw std::out_of_range("MixinVector index out of range");
            }
            void set(int i, T x) throw (std::out_of_range) {
                int size = int(self->size());
                if (i<0) i+= size;
                if (i>=0 && i<size)
                    (*self)[i] = x;
                else
                    throw std::out_of_range("MixinVector index out of range");
            }
        }
    };
    %enddef

    specialize_osg_MixinVector(bool, SwigMixin_GetBoolFromObj, Tcl_NewBooleanObj);
    specialize_osg_MixinVector(char, SwigMixinInt_As<char>,Tcl_NewIntObj);
    specialize_osg_MixinVector(int, Tcl_GetIntFromObj,Tcl_NewIntObj);
    specialize_osg_MixinVector(short, SwigMixinInt_As<short>, Tcl_NewIntObj);
    specialize_osg_MixinVector(long, SwigMixinInt_As<long>, Tcl_NewIntObj);
    specialize_osg_MixinVector(unsigned char, 
                          SwigMixinInt_As<unsigned char>, Tcl_NewIntObj);
    specialize_osg_MixinVector(unsigned int, 
                          SwigMixinInt_As<unsigned int>, Tcl_NewIntObj);
    specialize_osg_MixinVector(unsigned short, 
                          SwigMixinInt_As<unsigned short>, Tcl_NewIntObj);
    specialize_osg_MixinVector(unsigned long, 
                          SwigMixinInt_As<unsigned long>, Tcl_NewIntObj);
    specialize_osg_MixinVector(double, Tcl_GetDoubleFromObj, Tcl_NewDoubleObj);
    specialize_osg_MixinVector(float, SwigMixinDouble_As<float>, Tcl_NewDoubleObj);
    specialize_osg_MixinVector(std::string, 
                          SwigMixinString_AsString, SwigMixinString_FromString);

}
